#ifndef KMS_MODEL_ATTRIBUTE_LIST_H_
#define KMS_MODEL_ATTRIBUTE_LIST_H_

#include <stdbool.h>

#include "global_types.h"
#include "model/credentials.h"

typedef struct kms_attribute_t kms_attribute_t;

#define SEPCHAR_ATTR_PATH '/'
#define SEPCHAR_SYSFS_PATH '\0'

typedef struct kms_attribute_t
{
	//---- RIGHTS MGNT ----------------

	bool apply_credentials;

	kms_credentials_t credentials;

	//---- ATTRIBUTE LIST PTR ------------

	kms_attribute_t *next_in_list_ptr;
	kms_attribute_t *previous_in_list_ptr;

	/* in memory the attribute name is copied
	 * after syspath
	 * By exchanging the character used for separation
	 * (i.e. \0 or /) it is possible to switch between
	 * returning the syspath or the attribute path when
	 * accessing path variable
	 */
	char *sepchar_ptr;
	//---- ATTRIBUTE SYSPATH or ATTRIBUTE path -------------
	char path[];
} kms_attribute_t;

/**
 * Create an attribute and fill its structure
 *
 * \param attr_ptr return the pointer of the created attribute
 * \param sysfs_path the SYSFS path to the attribute
 * \param name of the attribute
 * \return returns RESULT_OK in case of no ERROR
 * 		- RESULT_NORESOURCES is returned in case of memory allocation issues
 */
error_code_t attribute_list_new_attribute_no_credentials(
		kms_attribute_t **attr_ptr, const char *sysfs_path, const char *name);

/**
 * Create an attribute and fill its structure
 *
 * \param attr_ptr return the pointer of the created attribute
 * \param sysfs_path the SYSFS path to the attribute
 * \param name of the attribute
 * \param credentials credentials struct or NULL in case no credentials needed
 * \return returns RESULT_OK in case of no ERROR
 * 		- RESULT_NORESOURCES is returned in case of memory allocation issues
 */
error_code_t attribute_list_new_attribute(kms_attribute_t **attr_ptr, const char *sysfs_path,
		const char *name, const kms_credentials_t *credentials);

/**
 * Free the memory of an attribute
 *
 * \param attribute pointer to the attribute
 * \return returns RESULT_OK in case of no ERROR
 * 		- RESULT_NORESOURCES is returned in case of memory allocation issues
 */
void attribute_list_free_attribute(kms_attribute_t *attribute);

/**
 * Called to add an sysfs attribute to the list of attributes we have to wait for.
 *
 * Use this function if you additionally want to set user and group ids as well as the access_mode of the
 * attribute once it appears.
 *
 * \param attribute to add to the list
 * \return returns RESULT_OK in case of no ERROR
 */
error_code_t attribute_list_add(kms_attribute_t *attribute);

/**
 * calls stat on the sysfs path of the attribute to check whether it is available or not.
 *
 * \param attribute the attribute to test for availability
 * \return true in case the attribute is available at the time of the request. false otherwise.
 */
bool attribute_list_is_attribute_available(kms_attribute_t *attribute);

/**
 * returns the first attribute in the list or NULL in case the list is empty
 * \return the first attribute in the list or NULL in case the list is empty
 */
kms_attribute_t *attribute_list_get_first_attribute(void);

/**
 * returns the attribute in the list located after the given one.
 * NULL is returned in case the given one is the last one in the list or NULL.
 * \param attribute the previous attribute of the requested one
 * \return the attriute next to the given one or NULl
 */
kms_attribute_t *attribute_list_get_next_attribute(kms_attribute_t *attribute);

/**
 * returns the first attribute in the list that starts with the given base sysfs path or NULL
 * in case no matching one is found.
 *
 * \param sysfs_path_base the base sysfs path to filter for
 * \return the first attribute in the list that starts with the given base sysfs path or NULL
 * in case no matching one is found.
 */
kms_attribute_t *attribute_list_get_first_attribute_filtered(const char *sysfs_path_base);

/**
 * returns the attribute in the list next to the given one that starts with the given base sysfs path or NULL
 * in case no matching one is found.
 *
 * \param sysfs_path_base the base sysfs path to filter for
 * \return the attribute next to the given one in the list that starts with the given base sysfs path or NULL
 * in case no matching one is found.
 */
kms_attribute_t *attribute_list_get_next_attribute_filtered(kms_attribute_t *attribute, const char *sysfs_path_base);

/**
 * This function removes the given attribute from the list of attributes to wait for and frees the memory allocated by
 * the data structure representing the attribute.
 * \param attribute the attribute to remove from the list. Be careful, the pointer becomes invalid after this call.
 */
void attribute_list_remove_and_free(kms_attribute_t *attribute);

/**
 * returns true in case the list of attributes is empty.
 * \return true in case the list is empty at the time of the call
 */
bool attribute_list_is_empty(void);

/**
 * call to empty the list
 */
void attribute_list_clear(void);

/**
 * Return string containing the sysfs path without the attribute name itself
 *
 * attribute_get_sysfs_path and attribute_get_attr_path must not be called
 * reentrant or in parallel for the same attribute.
 * Calling a second time before the first returned string is no longer needed
 * might cause the first string to change unexpectedly
 *
 * \param attribute pointer to the attribute
 * \return returns pointer to the string
 */
char* attribute_get_sysfs_path(kms_attribute_t *attribute);

/**
 * Return string containing the path to an attribute (incl. sysfs path)
 *
 * attribute_get_sysfs_path and attribute_get_attr_path must not be called
 * reentrant or in parallel for the same attribute.
 * Calling a second time before the first returned string is no longer needed
 * might cause the first string to change unexpectedly
 *
 * \param attribute pointer to the attribute
 * \return returns pointer to the string
 */
char* attribute_get_attr_path(kms_attribute_t *attribute);

#endif
